####################################################
# R functions for time series prediction
# Based on 
# - Farmer and Lafond (2016), Research Policy http://dx.doi.org/10.1016/j.respol.2015.11.001 
# - Lafond et al. (2018), Technological Forecasting and Social Change https://doi.org/10.1016/j.techfore.2017.11.001 
# Please cite [1] if you use MoorePredParam
# Please cite [2] if you use WrightPredParam

# This code is supplied with no warranty.
# François Lafond, December 2019. francois.lafond@inet.ox.ac.uk
####################################################


####################################################
# Moore's law
####################################################
# DF is a numeric vector
# TAU is the forecast horizon
# years must be consecutive, and ordered
MoorePredParam<-function(DF, TAU, theta=0.63){
  
  # Check for wrong inputs and return errors
  if( abs(theta)>=1 ){
    stop("theta should be less than 1 in absolute value")
  }
  
  # Compute parameters
  mu<-mean(diff(log(DF)))
  kk<-sd(diff(log(DF)))
  m<-length(DF)-1
  
  y.future<-log(DF[(m+1)])+mu*(1:TAU)
  # Create a sequence of variances of the point forecasts using Wright's model
  veps<-numeric(TAU)
  for(tau in 1:TAU){
    vv<- (1 + (2*theta*(m-1)/m) + theta^2) * (tau + (tau^2 / m))
    veps[tau]<-(kk^2/(1+theta^2)) * (-2*theta + vv )
  }
  
  # Return forecasts
  toreturn<-cbind(y.future,veps)
  colnames(toreturn)<-c("E(y)","Var(y)")
  return(toreturn)
  
}

####################################################
# Wright's law
####################################################
# DF is a dataframe with colums costs and experience
# e.future is future experience
# All values should be
# - in levels (not logs!) 
# - ordered, one per period.
# - no missing values
WrightPredParam<-function(DF, e.future, rho=0.19){
  
  # Check for wrong inputs and return errors
  if( class(DF)!="data.frame" ){
    stop("The first argument supplied must be a data.frame")
  }
  if( dim(DF)[2]!=2 ){
    stop("The first argument supplied must have exactly two columns")
  }
  colnames(DF)<-c("yy","ee")
  if( !is.numeric(DF$yy) |
      !is.numeric(DF$ee) | 
      !is.numeric(e.future) |
      !is.numeric(rho) ){
    stop("Some supplied values are not numeric")
  }
  if(any( c(is.na(DF),is.na(e.future)) )) {
    stop("Some supplied values are NA")
  }
  if( abs(rho)>=1 ){
    stop("The third argument rho should be less than 1 in absolute value")
  }
  
  # Useful shorthands
  TT<-dim(DF)[1]
  TAU<-length(e.future)
  
  # Compute parameters
  LM<-lm(diff(log(DF$yy))~diff(log(DF$ee))+0)
  omega<-LM$coef[1]
  sigmae<-summary(LM)$sigma
  
  # Useful data transformation
  difX<-diff(log(DF$ee))
  difXfuture<-diff(c(log(DF$ee[TT]), log(e.future)))
  
  # Create a sequence of future point forecasts using Wright's model
  y.future<- log(DF$yy[TT]) + omega*(log(e.future)-log(DF$ee[TT]))
  
  # Create a sequence of variances of the point forecasts using Wright's model
  veps<-numeric(TAU)
  for(tau in 1:TAU){
    Hj<- (-1)* ( sum(difXfuture[1:tau])/sum(difX^2) ) *difX
    hjx<-rep(NA,(TT-2))
    for(ss in 1:(TT-2)){
      hjx[ss]<-(Hj[ss]+rho*Hj[ss+1])^2
    }
    varepsfactor<-(rho*Hj[1])^2 + 
      sum(hjx)+
      (Hj[TT-1]+rho)^2 +
      (tau-1)*(1+rho)^2 +  1
    veps[tau]<-(sigmae^2/(1+rho^2))*varepsfactor
  }
  
  # Return forecasts
  toreturn<-cbind(y.future,veps)
  colnames(toreturn)<-c("E(y)","Var(y)")
  return(toreturn)
  
}

####################################################
# Compute forecast intervals based on paramters
####################################################
GetPreds<-function(pp,ci){
  pointforecasts<-exp(pp[,1])
  lb<-exp( qnorm(p=((1-ci)/2), mean=pp[,1],sd=sqrt(pp[,2])) )
  ub<-exp( qnorm(p=((1+ci)/2), mean=pp[,1],sd=sqrt(pp[,2])) )
  toreturn<-cbind(pointforecasts,lb,ub)
  colnames(toreturn)<-c("point_forecast","lower_bound","upper_bound")
  return(toreturn)
}




