####################################################
# Example of use of R functions for time series prediction
# Based on
# - [1] Farmer and Lafond (2016), Research Policy http://dx.doi.org/10.1016/j.respol.2015.11.001 
# - [2] Lafond et al. (2018), Technological Forecasting and Social Change https://doi.org/10.1016/j.techfore.2017.11.001 
# Please cite [1] if you use MoorePredParam
# Please cite [2] if you use WrightPredParam

# This code is supplied with no warranty.
# François Lafond, December 2019. francois.lafond@inet.ox.ac.uk
####################################################


rm(list = ls())
library(xtable)

# Run the file Forecasts_functions.R
# or source it (change address as appropriate)
source("/Users/mainuser/Library/Mobile Documents/com~apple~CloudDocs/0-Projects-Misc/Forecasting_readme/code/Forecasts_functions.R")

# download some data
DF<-read.csv("https://dl.dropboxusercontent.com/s/wpekjpli68dbmr5/Photovoltaics.csv?dl=0")
# For future ref, range(DF[,2]) is 1976-2016

# keep only cost on the first column, experience as second column
DF<-DF[,c(1,4)]
# Create a shorthand for the number of years
TT<-length(DF[,1])

# General parameters
tau<-9               # forecast horizon
pred_interval<-0.95  # coverage for the prediction interval
THETA<-0.23          # autocorrelation parameter for Moore's law
RHO<-0.19            # autocorrelation parameter for Wright's law

## Moore
pred_param_Moore<-MoorePredParam(DF[,1],tau,theta=THETA)
xtable(pred_param_Moore)

pred_pi_Moore<-GetPreds(pred_param_Moore,pred_interval)
xtable(pred_pi_Moore)

## Wright
r<-mean(diff(log(DF[,2])))
# One can put any value e.g. accelerated diffusion r<-0.5, or another sequence for E.FUTURE
E.FUTURE<-DF[TT,2]*exp(r*(1:tau))
pred_param_Wright<-WrightPredParam(DF,E.FUTURE,rho=RHO)
pred_pi_Wright<-GetPreds(pred_param_Wright,pred_interval)

## Plot
par(mar=c(3,4,3,2))
par(las=1)
plot(DF[,1],ylim=range(c(DF[,1],pred_pi_Moore)),
     xlim=c(1,TT+tau),pch=16,log="y",
     xaxt="n",ylab="",xlab="",
     main="PV module price in 2016 $/Wp")
axis(1,at=1:(TT+tau),1976:(2016+tau))
lines(c(rep(NA,TT),pred_pi_Moore[,1]),type="l",lwd=3)
lines(c(rep(NA,TT),pred_pi_Moore[,2]),type="l",lwd=1)
lines(c(rep(NA,TT),pred_pi_Moore[,3]),type="l",lwd=1)
lines(c(rep(NA,TT),pred_pi_Wright[,1]),type="l",lwd=3,col=2)
lines(c(rep(NA,TT),pred_pi_Wright[,2]),type="l",lwd=1,col=2)
lines(c(rep(NA,TT),pred_pi_Wright[,3]),type="l",lwd=1,col=2)
legend("bottomleft",col=c(1:2),legend=c("Moore","Wright"),lwd=2)
